import { Suspense } from "react"
import { notFound } from "next/navigation"
import { Header } from "@/components/header"
import { TVSeasonDetails } from "@/components/tv-season-details"
import { TVEpisodesList } from "@/components/tv-episodes-list"
import { Skeleton } from "@/components/ui/skeleton"
import Link from "next/link"
import { Button } from "@/components/ui/button"
import { ArrowLeft } from "lucide-react"

async function getTVShow(id: string) {
  try {
    const res = await fetch(`https://api.themoviedb.org/3/tv/${id}?api_key=${process.env.TMDB_API_KEY}`, {
      next: { revalidate: 3600 },
    })

    if (!res.ok) {
      return null
    }

    return res.json()
  } catch (err) {
    return null
  }
}

async function getSeasonDetails(tvId: string, seasonNumber: string) {
  try {
    const res = await fetch(
      `https://api.themoviedb.org/3/tv/${tvId}/season/${seasonNumber}?api_key=${process.env.TMDB_API_KEY}`,
      { next: { revalidate: 3600 } },
    )

    if (!res.ok) {
      console.error(`TMDB season fetch failed (${res.status}) for TV ${tvId} season ${seasonNumber}`)
      return null
    }

    return res.json()
  } catch (err) {
    console.error("TMDB season fetch error:", err)
    return null
  }
}

export default async function TVSeasonPage({
  params,
}: {
  params: { id: string; seasonNumber: string }
}) {
  const [tvShow, seasonDetails] = await Promise.all([
    getTVShow(params.id),
    getSeasonDetails(params.id, params.seasonNumber),
  ])

  if (!tvShow || !seasonDetails) {
    notFound()
  }

  return (
    <div className="min-h-screen bg-background">
      <Header />

      <div className="container mx-auto px-4 py-8">
        <div className="mb-6">
          <Button variant="ghost" asChild className="mb-4">
            <Link href={`/tv/${params.id}/seasons`}>
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to All Seasons
            </Link>
          </Button>

          <div className="mb-8">
            <h1 className="text-3xl font-bold mb-2">
              {tvShow.name} - {seasonDetails.name}
            </h1>
            <p className="text-muted-foreground">
              {seasonDetails.episodes?.length || 0} Episodes
              {seasonDetails.air_date && ` • Aired ${new Date(seasonDetails.air_date).getFullYear()}`}
            </p>
          </div>
        </div>

        <div className="space-y-8">
          <Suspense fallback={<Skeleton className="h-64 w-full" />}>
            <TVSeasonDetails season={seasonDetails} tvShow={tvShow} />
          </Suspense>

          <div>
            <h2 className="text-2xl font-bold mb-6">Episodes</h2>
            <Suspense fallback={<TVEpisodesListSkeleton />}>
              <TVEpisodesList episodes={seasonDetails.episodes || []} />
            </Suspense>
          </div>
        </div>
      </div>
    </div>
  )
}

function TVEpisodesListSkeleton() {
  return (
    <div className="space-y-4">
      {Array.from({ length: 10 }).map((_, i) => (
        <div key={i} className="flex space-x-4 p-4 border rounded-lg">
          <Skeleton className="w-24 h-16 rounded" />
          <div className="flex-1 space-y-2">
            <Skeleton className="h-5 w-3/4" />
            <Skeleton className="h-4 w-1/4" />
            <Skeleton className="h-4 w-full" />
          </div>
        </div>
      ))}
    </div>
  )
}

export async function generateMetadata({
  params,
}: {
  params: { id: string; seasonNumber: string }
}) {
  const [tvShow, seasonDetails] = await Promise.all([
    getTVShow(params.id),
    getSeasonDetails(params.id, params.seasonNumber),
  ])

  if (!tvShow || !seasonDetails) {
    return {
      title: "Season Not Found",
    }
  }

  return {
    title: `${tvShow.name} - ${seasonDetails.name} - MoviesWeb`,
    description: `Watch episodes from ${seasonDetails.name} of ${tvShow.name}`,
  }
}
